/**
 * \file: exchnd_utils.c
 *
 * Generic functions that are used either in backends or in collectors
 * but are not directly related to them.
 *
 * \component: exchndd
 *
 * \author: Frederic Berat (fberat@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <malloc.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/mman.h>
#include <sys/resource.h>

#include "exchndd.h"

#define PRIORITY_MIN 97

static int pool_size;
static void *processing_stack;
static pthread_attr_t attr;

/*
 * \func exchnd_print_error
 *
 * This function writes an error message as a kernel message. It provides a
 * printf interface to add parameter and it also adds the prefix "exchnd:"
 * to the output. The output to the kernel message is needed because this
 * is a daemon and error messages can't be printed to standard error.
 *
 * \param format Format string for the message to write
 * \param ... Further parameters
 */
/* PRQA: Lint Message 530: False positive, the va_start is initializing args */
/*lint -save -e530 */
void exchnd_print_error(const char *format, ...)
{
    char buffer[256];
    char temp[256];
    va_list args;
    int kmsg;

    kmsg = open("/dev/kmsg", O_WRONLY);

    if (kmsg >= 0) {
        int ret = 0;

        va_start(args, format);
        vsnprintf(temp, 256, format, args);
        va_end(args);
        snprintf(buffer, 256, "<1>exchndd: %s\n", temp);

        ret = write(kmsg, buffer, strlen(buffer));

        /* We may actually be in emergency, there will be no retrial */
        (void)ret;
        close(kmsg);
    }
}
/*lint -restore */

void show_new_pagefault_count(const char *logtext,
                              const char *allowed_maj,
                              const char *allowed_min)
{
    static int last_majflt = 0, last_minflt = 0;
    struct rusage usage;

    getrusage(RUSAGE_SELF, &usage);

    printf("%-30.30s: Page faults, Major:%ld (Allowed %s), "
           "Minor:%ld (Allowed %s)\n", logtext,
           usage.ru_majflt - last_majflt, allowed_maj,
           usage.ru_minflt - last_minflt, allowed_min);

    last_majflt = usage.ru_majflt;
    last_minflt = usage.ru_minflt;
}

static struct sched_param sparam;

void exchnd_init_scheduling(void)
{
    sched_getparam(0, &sparam);

    if (sparam.sched_priority < PRIORITY_MIN) {
        exchnd_print_error("Priority was set too low: %d\n",
                           sparam.sched_priority);
        sparam.sched_priority = sched_get_priority_max(SCHED_FIFO);
    }

    sched_setscheduler(0, SCHED_FIFO, &sparam);
}

int exchnd_create_thread(pthread_t *t, void *(*f)(void *), void *arg, int p)
{
    int ret = 0;

    if (p == EXH_NORMAL_TH) {
        pthread_attr_init(&attr);
        pthread_attr_setstacksize(&attr, PTHREAD_STACK_MIN);
    } else if (p == EXH_INIT_PROCESS_TH) {
        pthread_attr_init(&attr);
        pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
        pthread_attr_setschedpolicy(&attr, SCHED_FIFO);
        pthread_attr_setschedparam(&attr, &sparam);
        pthread_attr_setstack(&attr, processing_stack, pool_size);
        /* Thread shall release all resources (ptrace ...) on exit */
        pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    }

    ret = pthread_create(t, &attr, f, arg);

    if (p == EXH_NORMAL_TH)
        pthread_attr_destroy(&attr);

    return ret;
}

void exchnd_prepare_malloc(void)
{
    /* Turn off malloc trimming. */
    mallopt(M_TRIM_THRESHOLD, -1);
    /* Turn off mmap usage. */
    mallopt(M_MMAP_MAX, 0);
}

void exchnd_mem_rellock(void)
{
    /* We only lock current memory as we don't want to lock mapped memory
     * from remote processes.
     */
    if (mlockall(MCL_CURRENT))
        exchnd_print_error("Unable to lock memory Err=%s",
                           strerror(errno));
}

int exchnd_lock_memory(int memory_pool)
{
    exchnd_prepare_malloc();

    if (mlockall(MCL_CURRENT | MCL_FUTURE)) {
        exchnd_print_error("Unable to lock memory Err=%s",
                           strerror(errno));

        return EXIT_FAILURE;
    }

    if (posix_memalign(&processing_stack, sysconf(_SC_PAGESIZE), memory_pool)) {
        exchnd_print_error("exchnd processing_stack allocation failed. Err=%s",
                           strerror(errno));

        return EXIT_FAILURE;
    }

    /* Effectively lock each page off the memory processing_stack. */
    memset(processing_stack, 0, memory_pool);
    pool_size = memory_pool;

    return EXIT_SUCCESS;
}

void exchnd_unlock_memory(void)
{
    munlockall();

    free(processing_stack);
}

void exchnd_prepare_pool(void)
{
    char *pool = NULL;
    int i = 0;

    for (i = 2; i <= pool_size; i *= 2) {
        pool = malloc(i);

        if (!pool)
            break;

        memset(pool, 0, i);

        free(pool);
    }
}

void exchnd_init_list(struct msg_list *m)
{
    m->prev = m;
    m->next = m;
}

static void __exchnd_list_del(struct msg_list *p, struct msg_list *n)
{
    n->prev = p;
    p->next = n;
}

void exchnd_list_del(struct msg_list *m)
{
    __exchnd_list_del(m->prev, m->next);
    exchnd_init_list(m);
}

static void __exchnd_list_add(struct msg_list *m,
                              struct msg_list *p,
                              struct msg_list *n)
{
    n->prev = m;
    m->next = n;
    m->prev = p;
    p->next = m;
}

void exchnd_list_add_tail(struct msg_list *m, struct msg_list *h)
{
    __exchnd_list_add(m, h->prev, h);
}

/****************
 *  File accessors
 *  fopen had to be removed as it generates minor page faults.
 *  These page faults are not likely to be resolved in case of Out Of Memory.
 *  Then we had to create a FILE like structure that allows to get access to a
 *  file line by line.
 */

/**
 * \func exchnd_fgets
 *
 * Read at most size-1 characters from the given file.
 * Stop when a newline has been read, or the count runs out.
 * Return first argument, or NULL if no characters were read.
 *
 */
char *exchnd_fgets(char *buffer, int size, struct exchnd_file *file)
{
    char *nl;
    char *s;
    int fd = file->fd;
    int *len = &file->len;
    int curr_len = 0;
    char **curr = &file->curr;
    char *local = file->local;

    if ((size == 0) || (buffer == NULL))
        return NULL;

    /* Initializing buffer start pointer */
    s = buffer;
    /* Let some space for final '\0' */
    size--;

    while (size > 0) {
        if (*len <= 0) {
            /* Refill local string */
            int cnt = 0;
            memset(local, 0, MAX_SIZE);
            cnt = read(fd, local, MAX_SIZE - 1);

            if (cnt <= 0) {
                /* Nothing more to read. Did we found something previously ? */
                if (s == buffer)
                    return NULL;

                /* Then exit loop and return buffer */
                break;
            }

            /* Update pointers */
            *len = MAX_SIZE - 1;
            *curr = local;
            /* Avoid artifacts */
            local[cnt] = '\0';
        }

        curr_len = *len;

        if (curr_len > size)
            curr_len = size;

        nl = memchr((void *)*curr, '\n', curr_len);

        if (nl != NULL) {
            curr_len = nl + 1 - *curr;
            /* Copying the found line in the buffer */
            memcpy((void *)s, (void *)*curr, curr_len);
            /* Final character */
            s[curr_len] = '\0';
            /* Update pointers */
            *curr += curr_len;
            *len -= curr_len;
            return buffer;
        }

        memcpy((void *)s, (void *)*curr, curr_len);
        s += curr_len;
        size -= curr_len;
        /* Update pointers */
        *len -= curr_len;
        *curr += curr_len;
    }

    /* Final character */
    *s = 0;
    return buffer;
}

/*
 * \func exchnd_init_file
 *
 * Initialize exchnd file descriptor.
 * Firs zeroed local string so that we don't have artifacts.
 * Initialize current pointer and length.
 * Initialize the file descriptor to -1.
 */
void exchnd_init_file(struct exchnd_file *file)
{
    int *len = &file->len;
    char **curr = &file->curr;
    char *local = file->local;

    memset(local, 0, MAX_SIZE);
    *curr = local;
    *len = 0;

    file->fd = -1;
}

/*
 * \func exchnd_close
 *
 * Close a file descriptor and re-init the exchnd descriptor for potential
 * reuse.
 */
void exchnd_close(struct exchnd_file *file)
{
    close(file->fd);
    exchnd_init_file(file);
}
